<?php

namespace App\Services;

use App\Models\Payment;
use App\Models\Transaction;
use App\Models\Refund;
use Srmklive\PayPal\Services\PayPal as PayPalClient;
use Stripe\Stripe;
use Stripe\PaymentIntent;
use Razorpay\Api\Api as RazorpayApi;
use Exception;
use Illuminate\Support\Facades\Log;
use GuzzleHttp\Client;

class PaymentHandler
{
    protected $gateway;

    public function __construct($gateway)
    {
        $this->gateway = strtolower($gateway);
    }

    public function initiatePayment(Payment $payment)
    {
        try {
            $method = 'process' . ucfirst($this->gateway);
            if (method_exists($this, $method)) {
                return $this->$method($payment);
            }
            throw new Exception("Payment gateway '{$this->gateway}' is not supported.");
        } catch (Exception $e) {
            Log::error('Initiate Payment Error: ' . $e->getMessage());
            $payment->update(['Payment_status' => 'failed', 'Transaction_id' => null]);
            throw $e;
        }
    }

    public function verifyPayment($transactionId, $gatewayData = [])
    {
        try {
            $method = 'verify' . ucfirst($this->gateway);
            if (method_exists($this, $method)) {
                return $this->$method($transactionId, $gatewayData);
            }
            throw new Exception("Payment gateway '{$this->gateway}' verification not supported.");
        } catch (Exception $e) {
            Log::error('Verify Payment Error: ' . $e->getMessage());
            throw $e;
        }
    }

    public function processRefund(Payment $payment, Refund $refund)
    {
        $method = 'refund' . ucfirst($this->gateway);
        if (method_exists($this, $method)) {
            return $this->$method($payment, $refund);
        }
        throw new Exception("Refund not supported for gateway '{$this->gateway}'.");
    }

    protected function processPayPal(Payment $payment)
    {
        $provider = new PayPalClient;
        $provider->setApiCredentials(config('paypal'));
        $provider->getAccessToken();

        $response = $provider->createOrder([
            'intent' => 'CAPTURE',
            'purchase_units' => [
                [
                    'amount' => ['currency_code' => $payment->Currency, 'value' => $payment->Amount],
                    'description' => 'Payment for Subscription ID: ' . ($payment->Subscription_id ?? 'N/A'),
                ],
            ],
            'application_context' => [
                'return_url' => url('/api/payments/callback/paypal'),
                'cancel_url' => url('/api/payments/callback/paypal'),
            ],
        ]);

        if (isset($response['id']) && $response['status'] === 'CREATED') {
            $payment->update(['Payment_status' => 'pending', 'Transaction_id' => $response['id']]);
            foreach ($response['links'] as $link) {
                if ($link['rel'] === 'approve') {
                    return ['approval_url' => $link['href'], 'transaction_id' => $response['id']];
                }
            }
        }
        throw new Exception('PayPal payment creation failed');
    }

    protected function verifyPayPal($transactionId)
    {
        $provider = new PayPalClient;
        $provider->setApiCredentials(config('paypal'));
        $provider->getAccessToken();

        $response = $provider->capturePaymentOrder($transactionId);
        if (isset($response['status']) && $response['status'] === 'COMPLETED') {
            $payment = Payment::where('Transaction_id', $transactionId)->firstOrFail();
            $payment->update(['Payment_status' => 'completed']);

            Transaction::create([
                'Payment_id' => $payment->Payment_id,
                'Transaction_type' => 'payment',
                'Amount' => $payment->Amount,
                'Currency' => $payment->Currency,
                'Payment_method' => 'paypal',
                'Status' => 'completed',
                'Gateway_response' => json_encode($response),
                'Processed_at' => now(),
            ]);

            return true;
        }
        throw new Exception('PayPal verification failed');
    }

    protected function refundPayPal(Payment $payment, Refund $refund)
    {
        $provider = new PayPalClient;
        $provider->setApiCredentials(config('paypal'));
        $provider->getAccessToken();

        $client = new Client();
        $accessToken = $provider->getAccessToken();
        $response = $client->post('https://api-m.paypal.com/v2/payments/captures/' . $payment->Transaction_id . '/refund', [
            'headers' => [
                'Authorization' => 'Bearer ' . $accessToken['access_token'],
                'Content-Type' => 'application/json',
            ],
            'json' => [
                'amount' => [
                    'value' => $refund->Amount,
                    'currency_code' => $refund->Currency,
                ],
            ],
        ]);

        $result = json_decode((string)$response->getBody(), true);
        if (isset($result['status']) && $result['status'] === 'COMPLETED') {
            return $result;
        }
        throw new Exception('PayPal refund failed: ' . (isset($result['message']) ? $result['message'] : 'Unknown error'));
    }

    protected function processStripe(Payment $payment)
    {
        Stripe::setApiKey(config('stripe.secret'));
        $paymentIntent = PaymentIntent::create([
            'amount' => $payment->Amount * 100,
            'currency' => strtolower($payment->Currency),
            'description' => 'Payment for Subscription ID: ' . ($payment->Subscription_id ?? 'N/A'),
            'metadata' => ['payment_id' => $payment->Payment_id],
        ]);
        $payment->update(['Payment_status' => 'pending', 'Transaction_id' => $paymentIntent->id]);
        return ['client_secret' => $paymentIntent->client_secret, 'transaction_id' => $paymentIntent->id];
    }

    protected function verifyStripe($transactionId)
    {
        Stripe::setApiKey(config('stripe.secret'));
        $paymentIntent = PaymentIntent::retrieve($transactionId);
        if ($paymentIntent->status === 'succeeded') {
            $payment = Payment::where('Transaction_id', $transactionId)->firstOrFail();
            $payment->update(['Payment_status' => 'completed']);

            Transaction::create([
                'Payment_id' => $payment->Payment_id,
                'Transaction_type' => 'payment',
                'Amount' => $payment->Amount,
                'Currency' => $payment->Currency,
                'Payment_method' => 'stripe',
                'Status' => 'completed',
                'Gateway_response' => json_encode($paymentIntent),
                'Processed_at' => now(),
            ]);

            return true;
        }
        throw new Exception('Stripe verification failed');
    }

    protected function refundStripe(Payment $payment, Refund $refund)
    {
        Stripe::setApiKey(config('stripe.secret'));
        $response = \Stripe\Refund::create([
            'payment_intent' => $payment->Transaction_id,
            'amount' => $refund->Amount * 100,
            'currency' => strtolower($refund->Currency),
        ]);

        if ($response->status === 'succeeded') {
            return $response;
        }
        throw new Exception('Stripe refund failed');
    }

    protected function processRazorpay(Payment $payment)
    {
        $razorpayKey = config('razorpay.key');
        $razorpaySecret = config('razorpay.secret');
        Log::info('Razorpay Process - Key: ' . $razorpayKey . ', Secret: ' . $razorpaySecret);

        if (!$razorpayKey || !$razorpaySecret) {
            throw new Exception('Razorpay keys missing. Check .env and config.');
        }

        try {
            $api = new RazorpayApi($razorpayKey, $razorpaySecret);
            $order = $api->order->create([
                'amount' => $payment->Amount * 100,
                'currency' => 'INR',
                'receipt' => 'order_' . $payment->Payment_id,
                'notes' => ['user_id' => $payment->User_id, 'subscription_id' => $payment->Subscription_id],
            ]);

            if ($order && isset($order->id)) {
                $payment->update(['Payment_status' => 'pending', 'Transaction_id' => $order->id]);
                Log::info('Razorpay Order Created: ' . $order->id);
                return ['order_id' => $order->id, 'transaction_id' => $order->id];
            }
            throw new Exception('Razorpay order creation failed');
        } catch (Exception $e) {
            Log::error('Razorpay API Error: ' . $e->getMessage());
            throw new Exception('Razorpay payment creation failed: ' . $e->getMessage());
        }
    }

    protected function verifyRazorpay($transactionId, $gatewayData)
    {
        $razorpayKey = config('razorpay.key');
        $razorpaySecret = config('razorpay.secret');
        Log::info('Razorpay Verify - Key: ' . $razorpayKey . ', Secret: ' . $razorpaySecret);

        if (!$razorpayKey || !$razorpaySecret) {
            throw new Exception('Razorpay keys missing. Check .env and config.');
        }

        try {
            $api = new RazorpayApi($razorpayKey, $razorpaySecret);
            $attributes = [
                'razorpay_order_id' => $transactionId,
                'razorpay_payment_id' => $gatewayData['razorpay_payment_id'],
                'razorpay_signature' => $gatewayData['razorpay_signature'],
            ];
            $api->utility->verifyPaymentSignature($attributes);
            $payment = Payment::where('Transaction_id', $transactionId)->firstOrFail();
            $payment->update(['Payment_status' => 'completed']);

            Transaction::create([
                'Payment_id' => $payment->Payment_id,
                'Transaction_type' => 'payment',
                'Amount' => $payment->Amount,
                'Currency' => $payment->Currency,
                'Payment_method' => 'razorpay',
                'Status' => 'completed',
                'Gateway_response' => json_encode($gatewayData),
                'Processed_at' => now(),
            ]);

            return true;
        } catch (Exception $e) {
            Log::error('Razorpay Verify Error: ' . $e->getMessage());
            throw new Exception('Razorpay verification failed: ' . $e->getMessage());
        }
    }

    protected function refundRazorpay(Payment $payment, Refund $refund)
    {
        $razorpayKey = config('razorpay.key');
        $razorpaySecret = config('razorpay.secret');
        $api = new RazorpayApi($razorpayKey, $razorpaySecret);

        $payment = Payment::findOrFail($payment->Payment_id);
        $response = $api->payment->fetch($payment->Transaction_id)->refund([
            'amount' => $refund->Amount * 100, // Amount in paise
            'speed' => 'optimum',
        ]);

        if ($response->status === 'processed') {
            return $response;
        }
        throw new Exception('Razorpay refund failed');
    }
}
